package explorviz.hpc_monitoring.filter.reconstruction;

import java.io.Serializable;
import java.util.*;
import explorviz.hpc_monitoring.record.Trace;
import explorviz.hpc_monitoring.record.TraceMetadata;
import explorviz.hpc_monitoring.record.events.AbstractOperationEvent;
import explorviz.hpc_monitoring.record.events.normal.*;

public class TraceBuffer {
    private static final Comparator<AbstractOperationEvent> COMPARATOR          = new AbstractOperationEventComperator();

    private TraceMetadata                                   traceMetadata;
    private final TreeSet<AbstractOperationEvent>           events              = new TreeSet<AbstractOperationEvent>(
                                                                                        COMPARATOR);

    private boolean                                         closeable;
    private boolean                                         damaged;

    private int                                             openEvents;
    private int                                             maxOrderIndex       = -1;
    private long                                            maxLoggingTimestamp = -1;

    public final long getMaxLoggingTimestamp() {
        return maxLoggingTimestamp;
    }

    public final void insertEvent(final AbstractOperationEvent event) {
        setMaxLoggingTimestamp(event);
        final int orderIndex = setMaxOrderIndex(event);

        if (event instanceof BeforeOperationEvent) {
            if (orderIndex == 0) {
                closeable = true;
            }
            openEvents++;
        }
        else if (event instanceof AfterOperationEvent) {
            openEvents--;
        }
        else if (event instanceof AfterFailedOperationEvent) {
            openEvents--;
        }

        if (!events.add(event)) {
            System.out.println("Duplicate entry for orderIndex " + orderIndex
                    + " with traceId " + traceMetadata.getTraceId());
            damaged = true;
        }
    }

    private final void setMaxLoggingTimestamp(final AbstractOperationEvent event) {
        final long loggingTimestamp = event.getLoggingTimestamp();
        if (loggingTimestamp > maxLoggingTimestamp) {
            maxLoggingTimestamp = loggingTimestamp;
        }
    }

    private final int setMaxOrderIndex(final AbstractOperationEvent event) {
        final int orderIndex = event.getOrderIndex();
        if (orderIndex > maxOrderIndex) {
            maxOrderIndex = orderIndex;
        }
        return orderIndex;
    }

    public void setTrace(final TraceMetadata trace) {
        if (traceMetadata != null) {
            System.out.println("Duplicate Trace entry for traceId "
                    + trace.getTraceId());
            damaged = true;
            return;
        }
        traceMetadata = trace;
    }

    public final boolean isFinished() {
        return !isInvalid() && closeable;
    }

    public final boolean isInvalid() {
        return (((maxOrderIndex + 1) != events.size()) || events.isEmpty()
                || (openEvents != 0) || (traceMetadata == null) || damaged);
    }

    public final Trace toTrace() { // TODO still slow?
        final AbstractOperationEvent[] arrayEvents = new AbstractOperationEvent[events
                .size()];
        final Iterator<AbstractOperationEvent> iterator = events.iterator();
        int index = 0;
        while (iterator.hasNext()) {
            arrayEvents[index] = iterator.next();
            index++;
        }

        return new Trace(traceMetadata, arrayEvents);
    }

    /**
     * @author Jan Waller
     */
    private static final class AbstractOperationEventComperator implements
            Comparator<AbstractOperationEvent>, Serializable {
        private static final long serialVersionUID = 8920737343446332517L;

        @Override
        public int compare(final AbstractOperationEvent o1,
                final AbstractOperationEvent o2) {
            return o1.getOrderIndex() - o2.getOrderIndex();
        }
    }
}
