package explorviz.hpc_monitoring.reader;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.lmax.disruptor.EventHandler;
import com.lmax.disruptor.RingBuffer;
import com.lmax.disruptor.dsl.Disruptor;

import explorviz.hpc_monitoring.byteaccess.UnsafeBits;
import explorviz.hpc_monitoring.disruptor.ByteArrayEvent;
import explorviz.hpc_monitoring.disruptor.RecordEvent;
import explorviz.hpc_monitoring.filter.counting.CountingThroughputFilter;
import explorviz.hpc_monitoring.filter.reconstruction.TraceReconstructionFilter;
import explorviz.hpc_monitoring.record.IRecord;
import explorviz.hpc_monitoring.record.TraceMetadata;
import explorviz.hpc_monitoring.record.events.normal.AfterFailedOperationEvent;
import explorviz.hpc_monitoring.record.events.normal.AfterOperationEvent;
import explorviz.hpc_monitoring.record.events.normal.BeforeOperationEvent;
import gnu.trove.map.hash.TIntObjectHashMap;

public class MessageDistributer implements EventHandler<ByteArrayEvent> {

	private static final CountingThroughputFilter counter = new CountingThroughputFilter(
			"Records per second");

	private final TIntObjectHashMap<String> stringRegistry = new TIntObjectHashMap<String>(
			64);

	private final List<byte[]> waitingForStringMessages = new ArrayList<byte[]>(
			1024);

	private byte[] unreadBytes = null;

	private final RingBuffer<RecordEvent> ringBuffer;

	@SuppressWarnings("unchecked")
	public MessageDistributer(final EventHandler<RecordEvent> endReceiver) {
		final ExecutorService exec = Executors.newCachedThreadPool();
		final Disruptor<RecordEvent> disruptor = new Disruptor<RecordEvent>(
				RecordEvent.EVENT_FACTORY, 32768, exec);

		final EventHandler<RecordEvent>[] eventHandlers = new EventHandler[1];
		eventHandlers[0] = new TraceReconstructionFilter(
				1 * 1000 * 1000 * 1000, endReceiver);
		disruptor.handleEventsWith(eventHandlers);
		ringBuffer = disruptor.start();
	}

	@Override
	public void onEvent(final ByteArrayEvent event, final long sequence,
			final boolean endOfBatch) throws Exception {
		final byte[] received = event.getValue();
		final int receivedLength = event.getLength();

		byte[] messages = received;
		int messagesLength = receivedLength;

		if (unreadBytes != null) {
			final int unreadBytesLength = unreadBytes.length;

			messagesLength += unreadBytesLength;
			messages = new byte[messagesLength];

			System.arraycopy(unreadBytes, 0, messages, 0, unreadBytesLength);
			System.arraycopy(received, 0, messages, unreadBytesLength,
					receivedLength);
		}

		unreadBytes = messagesfromByteArray(messages, messagesLength);
	}

	private byte[] messagesfromByteArray(final byte[] b, final int readSize) {
		int offset = 0;

		while (offset < readSize) {
			if ((readSize - offset) < 4) {
				return createUnreadBytesArray(b, readSize, offset, false);
			}

			final int clazzId = UnsafeBits.getInt(b, offset);
			offset += 4;

			switch (clazzId) {
				case TraceMetadata.CLAZZ_ID: {
					if ((readSize - offset) < TraceMetadata.BYTE_LENGTH) {
						return createUnreadBytesArray(b, readSize, offset, true);
					}

					offset = readInTraceMetadata(b, offset);
					break;
				}
				case BeforeOperationEvent.CLAZZ_ID: {
					if ((readSize - offset) < BeforeOperationEvent.BYTE_LENGTH) {
						return createUnreadBytesArray(b, readSize, offset, true);
					}

					offset = readInBeforeOperationEvent(b, offset);
					break;
				}
				case AfterFailedOperationEvent.CLAZZ_ID: {
					if ((readSize - offset) < AfterFailedOperationEvent.BYTE_LENGTH) {
						return createUnreadBytesArray(b, readSize, offset, true);
					}

					offset = readInAfterFailedOperationEvent(b, offset);
					break;
				}
				case AfterOperationEvent.CLAZZ_ID: {
					if ((readSize - offset) < AfterOperationEvent.BYTE_LENGTH) {
						return createUnreadBytesArray(b, readSize, offset, true);
					}

					offset = readInAfterOperationEvent(b, offset);
					break;
				}
				case 4: {
					if ((readSize - offset) < (4 + 4)) {
						return createUnreadBytesArray(b, readSize, offset, true);
					}

					final int mapId = UnsafeBits.getInt(b, offset);
					offset += 4;
					final int stringLength = UnsafeBits.getInt(b, offset);
					offset += 4;

					if ((readSize - offset) < stringLength) {
						return createUnreadBytesArray(b, readSize, offset - 8,
								true);
					}

					final byte[] stringBytes = new byte[stringLength];
					System.arraycopy(b, offset, stringBytes, 0, stringLength);
					final String string = new String(stringBytes);
					offset += stringLength;

					addToRegistry(mapId, string);

					break;
				}
				default: {
					System.out.println("unknown class id " + clazzId
							+ " at offset " + (offset - 4));
					return null;
				}
			}
		}

		return null;
	}

	private int readInTraceMetadata(final byte[] b, int offset) {
		final long traceId = UnsafeBits.getLong(b, offset);
		offset += 8;
		final int hostnameId = UnsafeBits.getInt(b, offset);
		offset += 4;
		final long parentTraceId = UnsafeBits.getLong(b, offset);
		offset += 8;
		final int parentOrderId = UnsafeBits.getInt(b, offset);
		offset += 4;
		final int applicationId = UnsafeBits.getInt(b, offset);
		offset += 4;

		final String hostname = getStringFromRegistry(hostnameId);
		final String application = getStringFromRegistry(applicationId);

		if ((hostname != null) && (application != null)) {
			putInRingBuffer(new TraceMetadata(traceId, hostname, parentTraceId,
					parentOrderId, application));
		} else {
			final byte[] message = new byte[TraceMetadata.BYTE_LENGTH_WITH_CLAZZ_ID];
			System.arraycopy(b, offset
					- TraceMetadata.BYTE_LENGTH_WITH_CLAZZ_ID, message, 0,
					TraceMetadata.BYTE_LENGTH_WITH_CLAZZ_ID);
			putInWaitingMessages(message);
		}
		return offset;
	}

	private int readInBeforeOperationEvent(final byte[] b, int offset) {
		final long timestamp = UnsafeBits.getLong(b, offset);
		offset += 8;
		final long traceId = UnsafeBits.getLong(b, offset);
		offset += 8;
		final int orderIndex = UnsafeBits.getInt(b, offset);
		offset += 4;
		final int operationId = UnsafeBits.getInt(b, offset);
		offset += 4;

		final String operation = getStringFromRegistry(operationId);

		if (operation != null) {
			putInRingBuffer(new BeforeOperationEvent(timestamp, traceId,
					orderIndex, operation));
		} else {
			final byte[] message = new byte[BeforeOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID];
			System.arraycopy(b, offset
					- BeforeOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID, message,
					0, BeforeOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID);
			putInWaitingMessages(message);
		}
		return offset;
	}

	private int readInAfterFailedOperationEvent(final byte[] b, int offset) {
		final long timestamp = UnsafeBits.getLong(b, offset);
		offset += 8;
		final long traceId = UnsafeBits.getLong(b, offset);
		offset += 8;
		final int orderIndex = UnsafeBits.getInt(b, offset);
		offset += 4;
		final int operationId = UnsafeBits.getInt(b, offset);
		offset += 4;
		final int causeId = UnsafeBits.getInt(b, offset);
		offset += 4;

		final String operation = getStringFromRegistry(operationId);
		final String cause = getStringFromRegistry(causeId);

		if ((operation != null) && (cause != null)) {
			putInRingBuffer(new AfterFailedOperationEvent(timestamp, traceId,
					orderIndex, operation, cause));
		} else {
			final byte[] message = new byte[AfterFailedOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID];
			System.arraycopy(b, offset
					- AfterFailedOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID,
					message, 0,
					AfterFailedOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID);
			putInWaitingMessages(message);
		}
		return offset;
	}

	private int readInAfterOperationEvent(final byte[] b, int offset) {
		final long timestamp = UnsafeBits.getLong(b, offset);
		offset += 8;
		final long traceId = UnsafeBits.getLong(b, offset);
		offset += 8;
		final int orderIndex = UnsafeBits.getInt(b, offset);
		offset += 4;
		final int operationId = UnsafeBits.getInt(b, offset);
		offset += 4;

		final String operation = getStringFromRegistry(operationId);
		if (operation != null) {
			putInRingBuffer(new AfterOperationEvent(timestamp, traceId,
					orderIndex, operation));
		} else {
			final byte[] message = new byte[AfterOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID];
			System.arraycopy(b, offset
					- AfterOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID, message,
					0, AfterOperationEvent.BYTE_LENGTH_WITH_CLAZZ_ID);
			putInWaitingMessages(message);
		}

		return offset;
	}

	private void putInWaitingMessages(final byte[] message) {
		waitingForStringMessages.add(message);
	}

	private byte[] createUnreadBytesArray(final byte[] b, final int readSize,
			int offset, final boolean withClazzId) {
		if (withClazzId) {
			offset -= 4;
		}
		final int unreadBytesSize = readSize - offset;
		final byte[] unreadBytes = new byte[unreadBytesSize];
		System.arraycopy(b, offset, unreadBytes, 0, unreadBytesSize);
		return unreadBytes;
	}

	private void putInRingBuffer(final IRecord record) {
		counter.inputObjects(record);
		final long hiseq = ringBuffer.next();
		final RecordEvent valueEvent = ringBuffer.get(hiseq);
		valueEvent.setValue(record);
		ringBuffer.publish(hiseq);
	}

	public void addToRegistry(final int key, final String value) {
		stringRegistry.put(key, value);

		// System.out.println("put key " + key + " value " + value);

		checkWaitingMessages();
	}

	private void checkWaitingMessages() {
		final List<byte[]> localWaitingList = new ArrayList<byte[]>();
		for (final byte[] waitingMessage : waitingForStringMessages) {
			localWaitingList.add(waitingMessage);
		}
		waitingForStringMessages.clear();

		for (final byte[] waitingMessage : localWaitingList) {
			final int waitingMessageClazzId = UnsafeBits.getInt(waitingMessage,
					0);
			switch (waitingMessageClazzId) {
				case TraceMetadata.CLAZZ_ID:
					readInTraceMetadata(waitingMessage, 4);
					break;
				case BeforeOperationEvent.CLAZZ_ID:
					readInBeforeOperationEvent(waitingMessage, 4);
					break;
				case AfterFailedOperationEvent.CLAZZ_ID:
					readInAfterFailedOperationEvent(waitingMessage, 4);
					break;
				case AfterOperationEvent.CLAZZ_ID:
					readInAfterOperationEvent(waitingMessage, 4);
					break;
				default:
					break;
			}
		}
	}

	private String getStringFromRegistry(final int id) {
		return stringRegistry.get(id);
	}
}
