package explorviz.hpc_monitoring.reader;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import com.lmax.disruptor.EventHandler;
import com.lmax.disruptor.RingBuffer;
import com.lmax.disruptor.dsl.Disruptor;

import explorviz.hpc_monitoring.disruptor.ByteArrayEvent;
import explorviz.hpc_monitoring.disruptor.RecordEvent;

public final class TCPReader {
	private static final int MESSAGE_BUFFER_SIZE = 65536;

	private final int listeningPort;

	private ServerSocketChannel serversocket;
	private boolean active = true;

	private final RingBuffer<ByteArrayEvent> ringBuffer;

	private final ByteBuffer buffer;

	public TCPReader(final int listeningPort,
			final EventHandler<RecordEvent> endReceiver)
			throws IllegalArgumentException {
		this.listeningPort = listeningPort;

		buffer = ByteBuffer.allocateDirect(MESSAGE_BUFFER_SIZE);

		final ExecutorService exec = Executors.newCachedThreadPool();
		final Disruptor<ByteArrayEvent> disruptor = new Disruptor<ByteArrayEvent>(
				ByteArrayEvent.EVENT_FACTORY, 4096, exec);

		@SuppressWarnings("unchecked")
		final EventHandler<ByteArrayEvent>[] eventHandlers = new EventHandler[1];
		eventHandlers[0] = new MessageDistributer(endReceiver);
		disruptor.handleEventsWith(eventHandlers);
		ringBuffer = disruptor.start();
	}

	public void read() {
		try {
			open();
			while (active) {
				// TODO only one connection!
				final SocketChannel socketChannel = serversocket.accept();
				int readBytes = 0;
				while ((readBytes = socketChannel.read(buffer)) != -1) {
					final byte[] messages = new byte[readBytes];
					buffer.flip();
					buffer.get(messages, 0, readBytes);
					putInRingBuffer(messages, readBytes);
					buffer.clear();
				}

				serversocket.close();
			}
		} catch (final IOException ex) {
			System.out.println("Error in read() " + ex.toString());
		} finally {
			try {
				serversocket.close();
			} catch (final IOException e) {
				System.out.println("Error in read()" + e.toString());
			}
		}
	}

	private void open() throws IOException {
		serversocket = ServerSocketChannel.open();
		serversocket.socket().bind(new InetSocketAddress(listeningPort));
		System.out.println("listening on port " + listeningPort);
	}

	private void putInRingBuffer(final byte[] messages, final int readBytes) {
		final long hiseq = ringBuffer.next();
		final ByteArrayEvent valueEvent = ringBuffer.get(hiseq);
		valueEvent.setValue(messages);
		valueEvent.setLength(readBytes);
		ringBuffer.publish(hiseq);
	}

	public void terminate(final boolean error) {
		System.out.println("Shutdown of TCPReader requested.");
		active = false;
	}
}
