/***************************************************************************
 * Copyright 2013 Kieker Project (http://kieker-monitoring.net)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package explorviz.hpc_monitoring.connector;

import java.io.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicInteger;
import kieker.analysis.IProjectContext;
import kieker.analysis.plugin.Bits;
import kieker.analysis.plugin.annotation.*;
import kieker.analysis.plugin.filter.AbstractFilterPlugin;
import kieker.common.configuration.Configuration;
import kieker.common.logging.Log;
import kieker.common.logging.LogFactory;
import kieker.common.record.IMonitoringRecord;
import com.rabbitmq.client.*;
import explorviz.hpc_monitoring.StringRegistryRecord;

/**
 * A plugin used for kieker in the cloud.
 * All incoming events are put into a RabbitMQ, but are also passed to an output
 * port that can be used for
 * testing purposes.
 * 
 * @author Santje Finke
 * 
 * @since 1.8
 * 
 */
@Plugin(description = "A filter that writes all incoming events into a specified queue on a specified RabbitMQServer", outputPorts = { @OutputPort(name = RabbitMQConnector.OUTPUT_PORT_NAME, eventTypes = { Object.class }, description = "Provides each incoming object") }, configuration = {
        @Property(name = RabbitMQConnector.CONFIG_PROPERTY_NAME_PROVIDER, defaultValue = "localhost"),
        @Property(name = RabbitMQConnector.CONFIG_PROPERTY_NAME_QUEUE, defaultValue = "master"),
        @Property(name = RabbitMQConnector.CONFIG_PROPERTY_PASSWORD, defaultValue = "guest"),
        @Property(name = RabbitMQConnector.CONFIG_PROPERTY_USER, defaultValue = "guest") })
public class RabbitMQConnector extends AbstractFilterPlugin {
    private static final int                         MESSAGE_BUFFER_SIZE            = 1500;

    /**
     * The name of the input port receiving the incoming events.
     */
    public static final String                       INPUT_PORT_NAME_INVALID_TRACES = "inputInvalidTraces";
    public static final String                       INPUT_PORT_NAME_VALID_TRACES   = "inputValidTraces";

    /**
     * The name of the output port passing the incoming events.
     */
    public static final String                       OUTPUT_PORT_NAME               = "relayedEvents";
    /**
     * The name of the property determining the address of the used Server.
     */
    public static final String                       CONFIG_PROPERTY_NAME_PROVIDER  = "providerUrl";
    /**
     * The name of the property determining the name of the Queue.
     */
    public static final String                       CONFIG_PROPERTY_NAME_QUEUE     = "queueName";

    /**
     * The username that is used to connect to a queue.
     */
    public static final String                       CONFIG_PROPERTY_USER           = "guest";
    /**
     * The password that is used to connect to a queue.
     */
    public static final String                       CONFIG_PROPERTY_PASSWORD       = "guest";

    private static final Log                         LOG                            = LogFactory
                                                                                            .getLog(RabbitMQConnector.class);

    private ConnectionFactory                        factory;
    private Connection                               connection;

    private final String                             providerUrl;
    private Channel                                  channel;
    private final String                             queue;

    private String                                   username                       = "guest";
    private String                                   password                       = "guest";

    private final byte[]                             validTracesMessages            = new byte[MESSAGE_BUFFER_SIZE];
    private int                                      validTracesMessagesOffset      = 0;

    private final byte[]                             invalidTracesMessages          = new byte[MESSAGE_BUFFER_SIZE];
    private int                                      invalidTracesMessagesOffset    = 0;

    private final ConcurrentHashMap<String, Integer> stringReg                      = new ConcurrentHashMap<String, Integer>(
                                                                                            16,
                                                                                            0.75f,
                                                                                            2);

    private final AtomicInteger                      stringRegIndex                 = new AtomicInteger(
                                                                                            0);

    public RabbitMQConnector(final Configuration configuration,
            final IProjectContext projectContext) {
        super(configuration, projectContext);
        providerUrl = configuration
                .getStringProperty(CONFIG_PROPERTY_NAME_PROVIDER);
        queue = configuration.getStringProperty(CONFIG_PROPERTY_NAME_QUEUE);
        username = configuration.getStringProperty(CONFIG_PROPERTY_USER);
        password = configuration.getStringProperty(CONFIG_PROPERTY_PASSWORD);
        createConnectionFactory(providerUrl);
        try {
            connect();
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    private void createConnectionFactory(final String provider) {
        factory = new ConnectionFactory();
        factory.setHost(provider);
        factory.setConnectionTimeout(0);
        factory.setUsername(username);
        factory.setPassword(password);
    }

    private void connect() throws IOException {
        connection = factory.newConnection();
        channel = connection.createChannel();
        channel.queueDeclare("validTracesMaster", false, false, false, null);
        channel.queueDeclare("invalidTracesMaster", false, false, false, null);
        channel.queueDeclare("registryRecordsMaster", false, false, false, null);
    }

    private void sendRegistryRecord(StringRegistryRecord record) {
        // TODO propagate to each new rabbitMQ queue
        final ByteArrayOutputStream boas = new ByteArrayOutputStream();

        ObjectOutputStream out;
        try {
            out = new ObjectOutputStream(boas);
            out.writeObject(record);
            out.close();

            byte[] message2 = boas.toByteArray();
            sendMessage(message2, "registryRecordsMaster");
        }
        catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * This method represents the input port of this filter.
     * 
     * @param event
     *            The next event.
     */
    @InputPort(name = INPUT_PORT_NAME_VALID_TRACES, eventTypes = { Object.class }, description = "Receives incoming objects to be forwarded to a queue")
    public final void inputValidTraces(final Object monitoringRecord) {
        try {
            // if (monitoringRecord instanceof IMonitoringRecord) {
            // byte[] message2 = toByteArray((IMonitoringRecord)
            // monitoringRecord);

            final ByteArrayOutputStream boas = new ByteArrayOutputStream();

            ObjectOutputStream out = new ObjectOutputStream(boas);
            out.writeObject(monitoringRecord);
            out.close();

            byte[] message2 = boas.toByteArray(); // TODO optimize

            System.arraycopy(message2, 0, validTracesMessages,
                    validTracesMessagesOffset, message2.length);
            validTracesMessagesOffset += message2.length;

            if ((validTracesMessagesOffset > (MESSAGE_BUFFER_SIZE - MESSAGE_BUFFER_SIZE))) { // TODO
                // unsafe
                // 200
                // ||
                // writeQueue.isEmpty()
                Bits.putInt(validTracesMessages, validTracesMessagesOffset, -1);
                sendMessage(validTracesMessages, "validTracesMaster");
                validTracesMessagesOffset = 0;
            }
        }
        catch (final IOException e) {
            LOG.error("Error sending record", e);
        }
    }

    /**
     * This method represents the input port of this filter.
     * 
     * @param event
     *            The next event.
     */
    @InputPort(name = INPUT_PORT_NAME_INVALID_TRACES, eventTypes = { Object.class }, description = "Receives incoming objects to be forwarded to a queue")
    public final void inputInvalidTraces(final Object monitoringRecord) {
        try {
            // if (monitoringRecord instanceof IMonitoringRecord) {
            // byte[] message2 = toByteArray((IMonitoringRecord)
            // monitoringRecord);

            final ByteArrayOutputStream boas = new ByteArrayOutputStream();

            ObjectOutputStream out = new ObjectOutputStream(boas);
            out.writeObject(monitoringRecord);
            out.close();

            byte[] message2 = boas.toByteArray(); // TODO optimize

            System.arraycopy(message2, 0, invalidTracesMessages,
                    invalidTracesMessagesOffset, message2.length);
            invalidTracesMessagesOffset += message2.length;

            if ((invalidTracesMessagesOffset > (MESSAGE_BUFFER_SIZE - MESSAGE_BUFFER_SIZE))) { // TODO
                // unsafe
                // 200
                // ||
                // writeQueue.isEmpty()
                Bits.putInt(invalidTracesMessages, invalidTracesMessagesOffset,
                        -1);
                sendMessage(invalidTracesMessages, "invalidTracesMaster");
                invalidTracesMessagesOffset = 0;
            }
        }
        catch (final IOException e) {
            LOG.error("Error sending record", e);
        }
    }

    @SuppressWarnings("unused")
    private byte[] toByteArray(final IMonitoringRecord monitoringRecord) {
        final Class<?>[] recordTypes = monitoringRecord.getValueTypes();

        int arraySize = 4 + 8;

        for (Class<?> recordType : recordTypes) {
            if (recordType == String.class) {
                arraySize += 4;
            }
            else if ((recordType == Integer.class) || (recordType == int.class)) {
                arraySize += 4;
            }
            else if ((recordType == Long.class) || (recordType == long.class)) {
                arraySize += 8;
            }
            else if ((recordType == Float.class) || (recordType == float.class)) {
                arraySize += 4;
            }
            else if ((recordType == Double.class)
                    || (recordType == double.class)) {
                arraySize += 8;
            }
            else if ((recordType == Byte.class) || (recordType == byte.class)) {
                arraySize += 1;
            }
            else if ((recordType == Short.class) || (recordType == short.class)) {
                arraySize += 2;
            }
            else if ((recordType == Boolean.class)
                    || (recordType == boolean.class)) {
                arraySize += 1;
            }
            else {
                arraySize += 1;
            }
        }

        byte[] result = new byte[arraySize];
        int offset = 0;

        Bits.putInt(result, offset, getIdForString(monitoringRecord.getClass()
                .getName()));
        offset += 4;

        Bits.putLong(result, offset, monitoringRecord.getLoggingTimestamp());
        offset += 8;
        final Object[] recordFields = monitoringRecord.toArray();

        for (int i = 0; i < recordFields.length; i++) {
            if (recordFields[i] == null) {
                if (recordTypes[i] == String.class) {
                    Bits.putInt(result, offset, getIdForString(""));
                    offset += 4;
                }
                else if ((recordTypes[i] == Integer.class)
                        || (recordTypes[i] == int.class)) {
                    Bits.putInt(result, offset, 0);
                    offset += 4;
                }
                else if ((recordTypes[i] == Long.class)
                        || (recordTypes[i] == long.class)) {
                    Bits.putLong(result, offset, 0L);
                    offset += 8;
                }
                else if ((recordTypes[i] == Float.class)
                        || (recordTypes[i] == float.class)) {
                    Bits.putFloat(result, offset, 0);
                    offset += 4;
                }
                else if ((recordTypes[i] == Double.class)
                        || (recordTypes[i] == double.class)) {
                    Bits.putDouble(result, offset, 0);
                    offset += 8;
                }
                else if ((recordTypes[i] == Byte.class)
                        || (recordTypes[i] == byte.class)) {
                    Bits.putByte(result, offset, (byte) 0);
                    offset += 1;
                }
                else if ((recordTypes[i] == Short.class)
                        || (recordTypes[i] == short.class)) {
                    Bits.putShort(result, offset, (short) 0);
                    offset += 2;
                }
                else if ((recordTypes[i] == Boolean.class)
                        || (recordTypes[i] == boolean.class)) {
                    Bits.putBoolean(result, offset, false);
                    offset += 1;
                }
                else {
                    LOG.warn("Record with unsupported recordField of type "
                            + recordFields[i].getClass());
                    Bits.putByte(result, offset, (byte) 0);
                    offset += 1;
                }
            }
            else if (recordFields[i] instanceof String) {
                Bits.putInt(result, offset,
                        getIdForString((String) recordFields[i]));
                offset += 4;
            }
            else if (recordFields[i] instanceof Integer) {
                Bits.putInt(result, offset, (Integer) recordFields[i]);
                offset += 4;
            }
            else if (recordFields[i] instanceof Long) {
                Bits.putLong(result, offset, (Long) recordFields[i]);
                offset += 8;
            }
            else if (recordFields[i] instanceof Float) {
                Bits.putFloat(result, offset, (Float) recordFields[i]);
                offset += 4;
            }
            else if (recordFields[i] instanceof Double) {
                Bits.putDouble(result, offset, (Double) recordFields[i]);
                offset += 8;
            }
            else if (recordFields[i] instanceof Byte) {
                Bits.putByte(result, offset, (Byte) recordFields[i]);
                offset += 1;
            }
            else if (recordFields[i] instanceof Short) {
                Bits.putShort(result, offset, (Short) recordFields[i]);
                offset += 2;
            }
            else if (recordFields[i] instanceof Boolean) {
                Bits.putBoolean(result, offset, (Boolean) recordFields[i]);
                offset += 1;
            }
            else {
                LOG.warn("Record with unsupported recordField of type "
                        + recordFields[i].getClass());
                Bits.putByte(result, offset, (byte) 0);
                offset += 1;
            }
        }

        return result;
    }

    public int getIdForString(String value) {
        Integer result = stringReg.get(value);

        if (result == null) {
            result = stringRegIndex.getAndIncrement();
            stringReg.put(value, result);
            sendRegistryRecord(new StringRegistryRecord(result, value));
        }

        return result;
    }

    private void sendMessage(final byte[] message, String queueName)
            throws IOException {

        synchronized (this) {
            if (!connection.isOpen() || !channel.isOpen()) {
                connect();
            }
            channel.basicPublish("", queueName, MessageProperties.BASIC,
                    message);
        }
    }

    protected final void cleanup() {
        disconnect();
    }

    private void disconnect() {
        try {
            if (channel != null) {
                channel.close();
            }
        }
        catch (final IOException e) {
            LOG.info("Error closing connection", e);
        }

        try {
            if (connection != null) {
                connection.close();
            }
        }
        catch (final IOException e) {
            LOG.info("Error closing connection", e);
        }
    }

    @Override
    public final String toString() {
        final StringBuilder sb = new StringBuilder(128);
        sb.append(super.toString());
        sb.append("; Channel: '");
        if (null != channel) {
            sb.append(channel.toString());
        }
        else {
            sb.append("null");
        }
        sb.append("'; Connection: '");
        if (null != connection) {
            sb.append(connection.toString());
        }
        else {
            sb.append("null");
        }
        sb.append('\'');
        return sb.toString();
    }

    @Override
    public Configuration getCurrentConfiguration() {
        final Configuration configuration = new Configuration();
        configuration.setProperty(CONFIG_PROPERTY_NAME_PROVIDER, providerUrl);
        configuration.setProperty(CONFIG_PROPERTY_NAME_QUEUE, queue);
        configuration.setProperty(CONFIG_PROPERTY_PASSWORD, password);
        configuration.setProperty(CONFIG_PROPERTY_USER, username);
        return configuration;
    }
}
