/***************************************************************************
 * Copyright 2013 Kieker Project (http://kieker-monitoring.net)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package explorviz.hpc_monitoring.plugin;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import kieker.analysis.IProjectContext;
import kieker.analysis.plugin.annotation.*;
import kieker.analysis.plugin.filter.AbstractFilterPlugin;
import kieker.common.configuration.Configuration;
import kieker.common.logging.Log;
import kieker.common.logging.LogFactory;
import kieker.common.record.flow.IFlowRecord;
import explorviz.hpc_monitoring.record.Trace;
import explorviz.hpc_monitoring.record.TraceEventRecords;
import explorviz.hpc_monitoring.record.events.*;

/**
 * @author Jan Waller
 * 
 * @since 1.6
 */
@Plugin(name = "Trace Reconstruction Filter (Event)", description = "Filter to reconstruct event based (flow) traces", outputPorts = {
        @OutputPort(name = EventRecordTraceReconstructionFilter.OUTPUT_PORT_NAME_TRACE_VALID, description = "Outputs valid traces", eventTypes = { TraceEventRecords.class }),
        @OutputPort(name = EventRecordTraceReconstructionFilter.OUTPUT_PORT_NAME_TRACE_INVALID, description = "Outputs traces missing crucial records", eventTypes = { TraceEventRecords.class }) }, configuration = {
        @Property(name = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_NAME_TIMEUNIT, defaultValue = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_VALUE_TIMEUNIT),
        @Property(name = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_NAME_MAX_TRACE_DURATION, defaultValue = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_VALUE_MAX_TIME),
        @Property(name = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_NAME_MAX_TRACE_TIMEOUT, defaultValue = EventRecordTraceReconstructionFilter.CONFIG_PROPERTY_VALUE_MAX_TIME) })
public final class EventRecordTraceReconstructionFilter extends
        AbstractFilterPlugin {
    /**
     * The name of the output port delivering the valid traces.
     */
    public static final String           OUTPUT_PORT_NAME_TRACE_VALID            = "validTraces";
    /**
     * The name of the output port delivering the invalid traces.
     */
    public static final String           OUTPUT_PORT_NAME_TRACE_INVALID          = "invalidTraces";
    /**
     * The name of the input port receiving the trace records.
     */
    public static final String           INPUT_PORT_NAME_TRACE_RECORDS           = "traceRecords";

    /**
     * The name of the property determining the time unit.
     */
    public static final String           CONFIG_PROPERTY_NAME_TIMEUNIT           = "timeunit";
    /**
     * The name of the property determining the maximal trace duration.
     */
    public static final String           CONFIG_PROPERTY_NAME_MAX_TRACE_DURATION = "maxTraceDuration";
    /**
     * The name of the property determining the maximal trace timeout.
     */
    public static final String           CONFIG_PROPERTY_NAME_MAX_TRACE_TIMEOUT  = "maxTraceTimeout";
    /**
     * The default value of the properties for the maximal trace duration and
     * timeout.
     */
    public static final String           CONFIG_PROPERTY_VALUE_MAX_TIME          = "9223372036854775807";                                    // String.valueOf(Long.MAX_VALUE)
    /**
     * The default value of the time unit property (nanoseconds).
     */
    public static final String           CONFIG_PROPERTY_VALUE_TIMEUNIT          = "NANOSECONDS";                                            // TimeUnit.NANOSECONDS.name()

    private static final Log             LOG                                     = LogFactory
                                                                                         .getLog(EventRecordTraceReconstructionFilter.class);

    private final TimeUnit               timeunit;
    private final long                   maxTraceDuration;
    private final long                   maxTraceTimeout;
    private final boolean                timeout;
    private long                         maxEncounteredLoggingTimestamp          = -1;

    private final Map<Long, TraceBuffer> traceId2trace;

    /**
     * Creates a new instance of this class using the given parameters.
     * 
     * @param configuration
     *            The configuration for this component.
     * @param projectContext
     *            The project context for this component.
     */
    public EventRecordTraceReconstructionFilter(
            final Configuration configuration,
            final IProjectContext projectContext) {
        super(configuration, projectContext);

        final String recordTimeunitProperty = projectContext
                .getProperty(IProjectContext.CONFIG_PROPERTY_NAME_RECORDS_TIME_UNIT);
        TimeUnit recordTimeunit;
        try {
            recordTimeunit = TimeUnit.valueOf(recordTimeunitProperty);
        }
        catch (final IllegalArgumentException ex) { // already caught in
                                                    // AnalysisController,
                                                    // should never happen
            LOG.warn(recordTimeunitProperty
                    + " is no valid TimeUnit! Using NANOSECONDS instead.");
            recordTimeunit = TimeUnit.NANOSECONDS;
        }
        timeunit = recordTimeunit;

        final String configTimeunitProperty = configuration
                .getStringProperty(CONFIG_PROPERTY_NAME_TIMEUNIT);
        TimeUnit configTimeunit;
        try {
            configTimeunit = TimeUnit.valueOf(configTimeunitProperty);
        }
        catch (final IllegalArgumentException ex) {
            LOG.warn(configTimeunitProperty
                    + " is no valid TimeUnit! Using inherited value of "
                    + timeunit.name() + " instead.");
            configTimeunit = timeunit;
        }

        maxTraceDuration = timeunit.convert(configuration
                .getLongProperty(CONFIG_PROPERTY_NAME_MAX_TRACE_DURATION),
                configTimeunit);
        maxTraceTimeout = timeunit.convert(configuration
                .getLongProperty(CONFIG_PROPERTY_NAME_MAX_TRACE_TIMEOUT),
                configTimeunit);
        timeout = !((maxTraceTimeout == Long.MAX_VALUE) && (maxTraceDuration == Long.MAX_VALUE));
        traceId2trace = new ConcurrentHashMap<Long, TraceBuffer>();
    }

    /**
     * This method is the input port for the new events for this filter.
     * 
     * @param record
     *            The new record to handle.
     */
    @InputPort(name = INPUT_PORT_NAME_TRACE_RECORDS, description = "Reconstruct traces from incoming flow records", eventTypes = {
            Trace.class, AbstractOperationEvent.class })
    public void newEvent(final IFlowRecord record) {
        final Long traceId;
        TraceBuffer traceBuffer;
        final long loggingTimestamp;
        if (record instanceof Trace) {
            traceId = ((Trace) record).getTraceId();
            traceBuffer = traceId2trace.get(traceId);
            if (traceBuffer == null) { // first record for this id!
                synchronized (this) {
                    traceBuffer = traceId2trace.get(traceId);
                    if (traceBuffer == null) { // NOCS (DCL)
                        traceBuffer = new TraceBuffer();
                        traceId2trace.put(traceId, traceBuffer);
                    }
                }
            }
            traceBuffer.setTrace((Trace) record);
            loggingTimestamp = -1;
        }
        else if (record instanceof AbstractOperationEvent) {
            traceId = ((AbstractOperationEvent) record).getTraceId();
            traceBuffer = traceId2trace.get(traceId);
            if (traceBuffer == null) { // first record for this id!
                synchronized (this) {
                    traceBuffer = traceId2trace.get(traceId);
                    if (traceBuffer == null) { // NOCS (DCL)
                        traceBuffer = new TraceBuffer();
                        traceId2trace.put(traceId, traceBuffer);
                    }
                }
            }
            traceBuffer.insertEvent((AbstractOperationEvent) record);
            loggingTimestamp = ((AbstractOperationEvent) record)
                    .getLoggingTimestamp();
        }
        else {
            return; // invalid type which should not happen due to the specified
                    // eventTypes
        }
        if (traceBuffer.isFinished()) {
            synchronized (this) { // has to be synchronized because of timeout
                                  // cleanup
                traceId2trace.remove(traceId);
            }
            super.deliver(OUTPUT_PORT_NAME_TRACE_VALID,
                    traceBuffer.toTraceEvents());
        }
        if (timeout) {
            synchronized (this) {
                // can we assume a rough order of logging timestamps? (yes,
                // except with DB reader)
                if (loggingTimestamp > maxEncounteredLoggingTimestamp) {
                    maxEncounteredLoggingTimestamp = loggingTimestamp;
                }
                processTimeoutQueue(maxEncounteredLoggingTimestamp);
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void terminate(final boolean error) {
        synchronized (this) {
            for (final Entry<Long, TraceBuffer> entry : traceId2trace
                    .entrySet()) {
                final TraceBuffer traceBuffer = entry.getValue();
                if (traceBuffer.isInvalid()) {
                    super.deliver(OUTPUT_PORT_NAME_TRACE_INVALID,
                            traceBuffer.toTraceEvents());
                }
                else {
                    super.deliver(OUTPUT_PORT_NAME_TRACE_VALID,
                            traceBuffer.toTraceEvents());
                }
            }
            traceId2trace.clear();
        }
    }

    // only called within synchronized! We assume timestamps >= 0
    private void processTimeoutQueue(final long timestamp) {
        final long duration = timestamp - maxTraceDuration;
        final long traceTimeout = timestamp - maxTraceTimeout;
        for (final Iterator<Entry<Long, TraceBuffer>> iterator = traceId2trace
                .entrySet().iterator(); iterator.hasNext();) {
            final TraceBuffer traceBuffer = iterator.next().getValue();
            if ((traceBuffer.getMaxLoggingTimestamp() <= traceTimeout) // long
                                                                       // time
                                                                       // no see
                    || (traceBuffer.getMinLoggingTimestamp() <= duration)) { // max
                                                                             // duration
                                                                             // is
                                                                             // gone
                if (traceBuffer.isInvalid()) {
                    super.deliver(OUTPUT_PORT_NAME_TRACE_INVALID,
                            traceBuffer.toTraceEvents());
                }
                else {
                    super.deliver(OUTPUT_PORT_NAME_TRACE_VALID,
                            traceBuffer.toTraceEvents());
                }
                iterator.remove();
            }
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Configuration getCurrentConfiguration() {
        final Configuration configuration = new Configuration();
        configuration.setProperty(CONFIG_PROPERTY_NAME_TIMEUNIT,
                timeunit.name());
        configuration.setProperty(CONFIG_PROPERTY_NAME_MAX_TRACE_DURATION,
                String.valueOf(maxTraceDuration));
        configuration.setProperty(CONFIG_PROPERTY_NAME_MAX_TRACE_TIMEOUT,
                String.valueOf(maxTraceTimeout));
        return configuration;
    }

    /**
     * The TraceBuffer is synchronized to prevent problems with concurrent
     * access.
     * 
     * @author Jan Waller
     */
    private static final class TraceBuffer {
        private static final Log                                LOG                 = LogFactory
                                                                                            .getLog(TraceBuffer.class);
        private static final Comparator<AbstractOperationEvent> COMPARATOR          = new TraceOperationComperator();

        private Trace                                           trace;
        private final SortedSet<AbstractOperationEvent>         events              = new TreeSet<AbstractOperationEvent>(
                                                                                            COMPARATOR);

        private boolean                                         closeable;
        private boolean                                         damaged;
        private int                                             openEvents;
        private int                                             maxOrderIndex       = -1;

        private long                                            minLoggingTimestamp = Long.MAX_VALUE;
        private long                                            maxLoggingTimestamp = -1;

        private long                                            traceId             = -1;

        /**
         * Creates a new instance of this class.
         */
        public TraceBuffer() {
            // default empty constructor
        }

        public void insertEvent(final AbstractOperationEvent record) {
            final long myTraceId = record.getTraceId();
            synchronized (this) {
                if (traceId == -1) {
                    traceId = myTraceId;
                }
                else if (traceId != myTraceId) {
                    LOG.error("Invalid traceId! Expected: " + traceId
                            + " but found: " + myTraceId + " in event "
                            + record.toString());
                    damaged = true;
                }
                final long loggingTimestamp = record.getLoggingTimestamp();
                if (loggingTimestamp > maxLoggingTimestamp) {
                    maxLoggingTimestamp = loggingTimestamp;
                }
                if (loggingTimestamp < minLoggingTimestamp) {
                    minLoggingTimestamp = loggingTimestamp;
                }
                final int orderIndex = record.getOrderIndex();
                if (orderIndex > maxOrderIndex) {
                    maxOrderIndex = orderIndex;
                }
                if (record instanceof BeforeOperationEvent) {
                    if (orderIndex == 0) {
                        closeable = true;
                    }
                    openEvents++;
                }
                else if (record instanceof AfterOperationEvent) {
                    openEvents--;
                }
                else if (record instanceof AfterFailedOperationEvent) {
                    openEvents--;
                }
                if (!events.add(record)) {
                    LOG.error("Duplicate entry for orderIndex " + orderIndex
                            + " with traceId " + myTraceId);
                    damaged = true;
                }
            }
        }

        public void setTrace(final Trace trace) {
            final long myTraceId = trace.getTraceId();
            synchronized (this) {
                if (traceId == -1) {
                    traceId = myTraceId;
                }
                else if (traceId != myTraceId) {
                    LOG.error("Invalid traceId! Expected: " + traceId
                            + " but found: " + myTraceId + " in trace "
                            + trace.toString());
                    damaged = true;
                }
                if (this.trace == null) {
                    this.trace = trace;
                }
                else {
                    LOG.error("Duplicate Trace entry for traceId " + myTraceId);
                    damaged = true;
                }
            }
        }

        public boolean isFinished() {
            synchronized (this) {
                return closeable && !isInvalid();
            }
        }

        public boolean isInvalid() {
            synchronized (this) {
                return (trace == null)
                        || damaged
                        || (openEvents != 0)
                        || (((maxOrderIndex + 1) != events.size()) || events
                                .isEmpty());
            }
        }

        public TraceEventRecords toTraceEvents() {
            synchronized (this) {
                return new TraceEventRecords(
                        trace,
                        events.toArray(new AbstractOperationEvent[events.size()]));
            }
        }

        public long getMaxLoggingTimestamp() {
            synchronized (this) {
                return maxLoggingTimestamp;
            }
        }

        public long getMinLoggingTimestamp() {
            synchronized (this) {
                return minLoggingTimestamp;
            }
        }

        /**
         * @author Jan Waller
         */
        private static final class TraceOperationComperator implements
                Comparator<AbstractOperationEvent> {
            public TraceOperationComperator() {}

            public int compare(final AbstractOperationEvent o1,
                    final AbstractOperationEvent o2) {
                return o1.getOrderIndex() - o2.getOrderIndex();
            }
        }
    }
}
