/***************************************************************************
 * Copyright 2013 Kieker Project (http://kieker-monitoring.net)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package explorviz.hpc_monitoring.reader;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import kieker.analysis.IProjectContext;
import kieker.analysis.plugin.annotation.*;
import kieker.analysis.plugin.reader.AbstractReaderPlugin;
import kieker.common.configuration.Configuration;
import kieker.common.logging.Log;
import kieker.common.logging.LogFactory;
import kieker.common.record.IMonitoringRecord;
import explorviz.hpc_monitoring.Bits;
import explorviz.hpc_monitoring.record.Trace;
import explorviz.hpc_monitoring.record.events.*;

/**
 * Reads monitoring records from the queue of an established RabbitMQ
 * connection.
 * 
 * @author Santje Finke
 * 
 * @since 1.8
 */
@Plugin(description = "A reader which reads records from a RabbitMQ queue", dependencies = "This plugin needs the file 'rabbitmq.client-*.jar'.", outputPorts = { @OutputPort(name = TCPReader.OUTPUT_PORT_NAME_RECORDS, eventTypes = { Object.class }, description = "Output Port of the JMSReader") }, configuration = {
        @Property(name = TCPReader.CONFIG_PROPERTY_NAME_PROVIDERURL, defaultValue = "localhost"),
        @Property(name = TCPReader.CONFIG_PROPERTY_PORT, defaultValue = "9876")

})
public final class TCPReader extends AbstractReaderPlugin {
    private static final int           MESSAGE_BUFFER_SIZE              = 65536;

    private final byte[]               messages                         = new byte[MESSAGE_BUFFER_SIZE];

    /** The name of the output port delivering the received records. */
    public static final String         OUTPUT_PORT_NAME_RECORDS         = "monitoringRecords";
    /** The name of the configuration determining the RabbitMQ provider URL. */
    public static final String         CONFIG_PROPERTY_NAME_PROVIDERURL = "mqProviderUrl";
    /** The port that is used to connect to a queue. */
    public static final String         CONFIG_PROPERTY_PORT             = "9876";

    static final Log                   LOG                              = LogFactory
                                                                                .getLog(TCPReader.class); // NOPMD
                                                                                                          // package
                                                                                                          // for
                                                                                                          // inner
                                                                                                          // class

    private final String               providerUrl;
    private final int                  port;

    private final CountDownLatch       cdLatch                          = new CountDownLatch(
                                                                                1);

    private final Map<Integer, String> stringRegistry                   = new ConcurrentHashMap<Integer, String>(
                                                                                16,
                                                                                0.75f,
                                                                                2);

    private ServerSocket               serversocket;
    private final boolean              active                           = true;

    /**
     * Creates a new instance of this class using the given parameters.
     * 
     * @param configuration
     *            The configuration used to initialize the whole reader. Keep in
     *            mind that the configuration should contain the following
     *            properties:
     *            <ul>
     *            <li>The property {@link #CONFIG_PROPERTY_NAME_PROVIDERURL},
     *            e.g. {@code localhost}
     *            <li>The property {@link #CONFIG_PROPERTY_NAME_QUEUE}, e.g.
     *            {@code queue1}
     *            <li>The property {@link #CONFIG_PROPERTY_PASSWORD}, e.g.
     *            {@code password}
     *            <li>The property {@link #CONFIG_PROPERTY_USER}, e.g.
     *            {@code username}
     *            <li>The property {@link #CONFIG_PROPERTY_PORT}, e.g.
     *            {@code port}
     *            </ul>
     * @param projectContext
     *            The project context for this component.
     * 
     * @throws IllegalArgumentException
     *             If one of the properties is empty.
     */
    public TCPReader(final Configuration configuration,
            final IProjectContext projectContext)
            throws IllegalArgumentException {
        super(configuration, projectContext);

        // Initialize the reader bases on the given configuration.
        providerUrl = configuration
                .getStringProperty(CONFIG_PROPERTY_NAME_PROVIDERURL);
        port = configuration.getIntProperty(CONFIG_PROPERTY_PORT);

        // registryConsumer = new RabbitMQRegistryConsumer(this, providerUrl,
        // "registryRecords", username, password, port);
        // registryConsumer.start();
    }

    /**
     * Creates a new instance of this class using the given parameters.
     * 
     * @param configuration
     *            The configuration used to initialize the whole reader. Keep in
     *            mind that the configuration should contain the following
     *            properties:
     *            <ul>
     *            <li>The property {@link #CONFIG_PROPERTY_NAME_PROVIDERURL},
     *            e.g. {@code localhost}
     *            <li>The property {@link #CONFIG_PROPERTY_NAME_QUEUE}, e.g.
     *            {@code queue1}
     *            <li>The property {@link #CONFIG_PROPERTY_PASSWORD}, e.g.
     *            {@code password}
     *            <li>The property {@link #CONFIG_PROPERTY_USER}, e.g.
     *            {@code username}
     *            </ul>
     * 
     * @throws IllegalArgumentException
     *             If one of the properties is empty.
     * 
     * @deprecated To be removed in Kieker 1.8.
     */
    @Deprecated
    public TCPReader(final Configuration configuration)
            throws IllegalArgumentException {
        this(configuration, null);
    }

    /**
     * A call to this method is a blocking call.
     * 
     * @return true if the method succeeds, false otherwise.
     */
    public boolean read() {
        boolean retVal = true;
        try {
            open();
            while (active) {
                // TODO only one connection!
                final Socket socket = serversocket.accept();
                BufferedInputStream bufferedInputStream = new BufferedInputStream(
                        socket.getInputStream(), MESSAGE_BUFFER_SIZE);
                int readSize = 0;
                int toReadOffset = 0;
                while ((readSize = bufferedInputStream.read(messages,
                        toReadOffset, MESSAGE_BUFFER_SIZE - toReadOffset)) != -1) {
                    byte[] unreadBytes = messagesfromByteArray(messages,
                            readSize + toReadOffset);
                    if (unreadBytes != null) {
                        toReadOffset = unreadBytes.length;
                        System.arraycopy(unreadBytes, 0, messages, 0,
                                toReadOffset);
                    }
                    else {
                        toReadOffset = 0;
                    }
                }

                socket.close();
            }
        }
        catch (final IOException ex) { // NOPMD NOCS
                                       // (IllegalCatchCheck)
            LOG.error("Error in read()", ex);
            retVal = false;
        }
        finally {
            try {
                serversocket.close();
            }
            catch (final IOException e) {
                LOG.error("Error in read()", e);
            }
        }
        return retVal;
    }

    private void open() throws IOException {
        serversocket = new ServerSocket(port);
    }

    private byte[] messagesfromByteArray(final byte[] b, int readSize) {
        int offset = 0;

        while (offset < readSize) {
            if ((readSize - offset) < 4) {
                return createUnreadBytesArray(b, readSize, offset, false);
            }

            final int clazzId = Bits.getInt(b, offset);
            offset += 4;

            switch (clazzId) {
                case 0: {
                    if ((readSize - offset) < (8 + 4 + 8 + 4)) {
                        return createUnreadBytesArray(b, readSize, offset, true);
                    }
                }
                case 1: {
                    if ((readSize - offset) < (8 + 8 + 4 + 4)) {
                        return createUnreadBytesArray(b, readSize, offset, true);
                    }
                }
                case 2: {
                    if ((readSize - offset) < (8 + 8 + 4 + 4 + 4)) {
                        return createUnreadBytesArray(b, readSize, offset, true);
                    }
                }
                case 3: {
                    if ((readSize - offset) < (8 + 8 + 4 + 4)) {
                        return createUnreadBytesArray(b, readSize, offset, true);
                    }
                }
                case 4: {
                    if ((readSize - offset) < (4 + 4)) {
                        return createUnreadBytesArray(b, readSize, offset, true);
                    }
                }
            }

            IMonitoringRecord record = null;

            switch (clazzId) {
                case 0: {
                    final long traceId = Bits.getLong(b, offset);
                    offset += 8;
                    final Integer hostnameId = Bits.getInt(b, offset);
                    offset += 4;
                    final long parentTraceId = Bits.getLong(b, offset);
                    offset += 8;
                    final int parentOrderId = Bits.getInt(b, offset);
                    offset += 4;

                    record = new Trace(traceId,
                            getStringFromRegistry(hostnameId), parentTraceId,
                            parentOrderId);
                    break;
                }
                case 1: {
                    final long timestamp = Bits.getLong(b, offset);
                    offset += 8;
                    final long traceId = Bits.getLong(b, offset);
                    offset += 8;
                    final int orderIndex = Bits.getInt(b, offset);
                    offset += 4;
                    final Integer operationId = Bits.getInt(b, offset);
                    offset += 4;

                    record = new BeforeOperationEvent(timestamp, traceId,
                            orderIndex, getStringFromRegistry(operationId));
                    break;
                }
                case 2: {
                    final long timestamp = Bits.getLong(b, offset);
                    offset += 8;
                    final long traceId = Bits.getLong(b, offset);
                    offset += 8;
                    final int orderIndex = Bits.getInt(b, offset);
                    offset += 4;
                    final Integer operationId = Bits.getInt(b, offset);
                    offset += 4;
                    final Integer causeId = Bits.getInt(b, offset);
                    offset += 4;

                    record = new AfterFailedOperationEvent(timestamp, traceId,
                            orderIndex, getStringFromRegistry(operationId),
                            getStringFromRegistry(causeId));
                    break;
                }
                case 3: {
                    final long timestamp = Bits.getLong(b, offset);
                    offset += 8;
                    final long traceId = Bits.getLong(b, offset);
                    offset += 8;
                    final int orderIndex = Bits.getInt(b, offset);
                    offset += 4;
                    final Integer operationId = Bits.getInt(b, offset);
                    offset += 4;

                    record = new AfterOperationEvent(timestamp, traceId,
                            orderIndex, getStringFromRegistry(operationId));
                    break;
                }
                case 4: {
                    final Integer mapId = Bits.getInt(b, offset);
                    offset += 4;
                    final int stringLength = Bits.getInt(b, offset);
                    offset += 4;

                    if ((readSize - offset) < stringLength) {
                        return createUnreadBytesArray(b, readSize, offset - 8,
                                true);
                    }

                    byte[] stringBytes = new byte[stringLength];
                    System.arraycopy(b, offset, stringBytes, 0, stringLength);
                    String string = new String(stringBytes);
                    offset += stringLength;

                    addToRegistry(mapId, string);

                    break;
                }
                default: {
                    LOG.error("unknown class id " + clazzId);
                }
            }

            if ((record != null)
                    && !super.deliver(OUTPUT_PORT_NAME_RECORDS, record)) {
                LOG.error("deliverRecord returned false");
            }
        }

        return null;
    }

    private byte[] createUnreadBytesArray(final byte[] b, int readSize,
            int offset, boolean withClazzId) {
        if (withClazzId) {
            offset -= 4;
        }
        final int unreadBytesSize = readSize - offset;
        final byte[] unreadBytes = new byte[unreadBytesSize];
        System.arraycopy(b, offset, unreadBytes, 0, unreadBytesSize);
        return unreadBytes;
    }

    final void unblock() { // NOPMD (package visible for inner class)
        cdLatch.countDown();
    }

    /**
     * {@inheritDoc}
     */
    public void terminate(final boolean error) {
        LOG.info("Shutdown of RabbitMQReader requested.");
        unblock();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Configuration getCurrentConfiguration() {
        final Configuration configuration = new Configuration();

        configuration
                .setProperty(CONFIG_PROPERTY_NAME_PROVIDERURL, providerUrl);

        return configuration;
    }

    public void addToRegistry(Integer key, String value) {
        stringRegistry.put(key, value);
        System.out.println(key + " " + value);

        synchronized (this) {
            notifyAll();
        }
    }

    private String getStringFromRegistry(Integer id) {
        String result = stringRegistry.get(id);
        while (result == null) {
            try {
                synchronized (this) {
                    System.out.println("waiting for " + id);
                    this.wait();
                }
            }
            catch (InterruptedException e) {
                e.printStackTrace();
            }
            result = stringRegistry.get(id);
        }

        return result;
    }
}
