/***************************************************************************
 * Copyright 2013 Kieker Project (http://kieker-monitoring.net)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package explorviz.hpc_monitoring.record;

import java.io.Serializable;

/**
 * Provides methods to calculate the minimum, maximum, average and the standard
 * deviation of numerical data.
 * 
 * @author Florian Biss
 * 
 * @since 1.8
 */
public final class RuntimeStatisticInformation implements Serializable {

    private static final long serialVersionUID = -1628273045707598143L;

    private long              count            = 0;
    private long              min              = Long.MAX_VALUE;
    private long              max              = Long.MIN_VALUE;
    // Using doubles to handle numbers above Integer.MAX_VALUE. Overflows are
    // much worse than losing precision here.
    private double            sum;
    private double            squareSum;

    public RuntimeStatisticInformation(final long runtime) {
        set(runtime);
    }

    public long getCount() {
        return count;
    }

    public long getMin() {
        return min;
    }

    public long getMax() {
        return max;
    }

    public long getAvg() {
        if (count > 0) {
            return (long) (sum / count);
        }
        else {
            return -1;
        }

    }

    public long getStandardDeviation() {
        if (count <= 2) {
            return -1;
        }
        else {
            final double variance = (squareSum - ((sum * sum) / count))
                    / (count - 1);
            return (long) Math.sqrt(variance);
        }

    }

    public double getSum() {
        return sum;
    }

    public double getSquareSum() {
        return squareSum;
    }

    public void insert(final long data) {

        count++;
        final double dataDouble = data;
        sum += dataDouble;
        squareSum += dataDouble * dataDouble;
        min = Math.min(data, min);
        max = Math.max(data, max);

    }

    public void set(final long data) {
        count = 1;
        // final double dataDouble = data;
        // sum = dataDouble;
        // squareSum = dataDouble * dataDouble;
        // max = data;
        // min = data;
    }

    public void merge(final RuntimeStatisticInformation statistics) {

        count += statistics.getCount();
        // sum += statistics.getSum();
        // squareSum += statistics.getSquareSum();
        // min = Math.min(statistics.getMin(), min);
        // max = Math.max(statistics.getMax(), max);

    }

    @Override
    public String toString() {
        return count + ":" + min + ":" + max + ":" + sum + ":" + squareSum;
    }

}
