/***************************************************************************
 * Copyright 2013 Kieker Project (http://kieker-monitoring.net)
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ***************************************************************************/

package explorviz.hpc_monitoring.record;

import java.io.Serializable;
import java.util.Arrays;
import explorviz.hpc_monitoring.record.events.AbstractOperationEvent;

public final class TraceEventRecords implements Serializable {
    private static final long                 serialVersionUID = 8589405631073291022L;

    private final Trace                       trace;
    private final AbstractOperationEvent[]    traceEvents;
    private final RuntimeStatisticInformation runtimeInformation;

    /**
     * Creates a new instance of this class using the given parameters.
     * 
     * @param trace
     *            The trace to be stored in this object.
     * @param traceEvents
     *            The trace events to be stored in this object.
     */
    public TraceEventRecords(final Trace trace,
            final AbstractOperationEvent[] traceEvents) { // NOPMD (stored
                                                          // directly)
        this.trace = trace;
        this.traceEvents = traceEvents;
        runtimeInformation = new RuntimeStatisticInformation(1); // TODO
    }

    public Trace getTrace() {
        return trace;
    }

    public AbstractOperationEvent[] getTraceOperations() {
        return traceEvents; // NOPMD (internal array exposed)
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder(64);
        sb.append(super.toString());
        sb.append("\n\tTrace: ");
        sb.append(trace);
        for (final AbstractOperationEvent traceEvent : traceEvents) {
            sb.append("\n\t");
            sb.append(traceEvent.getClass().getSimpleName());
            sb.append(": ");
            sb.append(traceEvent);
        }
        sb.append("\n\t");
        sb.append(runtimeInformation.getClass().getSimpleName());
        sb.append(": ");
        sb.append(runtimeInformation);
        sb.append('\n');
        return sb.toString();
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = (prime * result) + ((trace == null) ? 0 : trace.hashCode()); // NOCS
                                                                              // (?:)
        result = (prime * result) + Arrays.hashCode(traceEvents);
        return result;
    }

    @Override
    public boolean equals(final Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        final TraceEventRecords other = (TraceEventRecords) obj;
        if (trace == null) {
            if (other.trace != null) {
                return false;
            }
        }
        else if (!trace.equals(other.trace)) {
            return false;
        }
        if (!Arrays.equals(traceEvents, other.traceEvents)) {
            return false;
        }
        return true;
    }

    public RuntimeStatisticInformation getRuntime() {
        return runtimeInformation;
    }
}
